"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createUriConverter = void 0;
const vscode_uri_1 = require("vscode-uri");
function createUriConverter(documents) {
    const syncedDocumentUriToFileName = new Map();
    const syncedDocumentFileNameToUri = new Map();
    const encodeds = new Map();
    documents?.onDidOpen(({ document }) => {
        const fileName = uriToFileName(document.uri);
        syncedDocumentUriToFileName.set(document.uri, fileName);
        syncedDocumentFileNameToUri.set(fileName, document.uri);
    });
    documents?.onDidClose(e => {
        const fileName = syncedDocumentUriToFileName.get(e.document.uri);
        assert(fileName, 'fileName not found');
        syncedDocumentUriToFileName.delete(e.document.uri);
        syncedDocumentFileNameToUri.delete(fileName);
    });
    return {
        uriToFileName,
        fileNameToUri,
    };
    function uriToFileName(uri, parsed) {
        const syncedDocumentFileName = syncedDocumentUriToFileName.get(uri);
        if (syncedDocumentFileName) {
            return syncedDocumentFileName;
        }
        parsed ??= vscode_uri_1.URI.parse(uri);
        if (parsed.scheme === 'file') {
            return parsed.fsPath.replace(/\\/g, '/');
        }
        const encoded = encodeURIComponent(`${parsed.scheme}://${parsed.authority}`);
        encodeds.set(encoded, parsed);
        return `/${encoded}${parsed.path}`;
    }
    function fileNameToUri(fileName) {
        const syncedDocumentUri = syncedDocumentFileNameToUri.get(fileName);
        if (syncedDocumentUri) {
            return syncedDocumentUri;
        }
        for (const [encoded, uri] of encodeds) {
            const prefix = `/${encoded}`;
            if (fileName.startsWith(prefix)) {
                return vscode_uri_1.URI.from({
                    scheme: uri.scheme,
                    authority: uri.authority,
                    path: fileName.substring(prefix.length),
                }).toString();
            }
        }
        return vscode_uri_1.URI.file(fileName).toString();
    }
}
exports.createUriConverter = createUriConverter;
function assert(condition, message) {
    if (!condition) {
        throw new Error(message);
    }
}
//# sourceMappingURL=uri.js.map