/*****************************************************************************
 * Copyright (c) 2019, 2025 CEA LIST and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Initial API and implementation
 *  Pauline DEVILLE (CEA LIST) pauline.deville@cea.fr - Bug 570743
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - GL-9
 *****************************************************************************/

package org.eclipse.papyrus.model2doc.gmf.template2structure.internal.mapping;

import java.net.URL;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.URI;
import org.eclipse.gmf.runtime.diagram.core.preferences.PreferencesHint;
import org.eclipse.gmf.runtime.diagram.ui.image.ImageFileFormat;
import org.eclipse.gmf.runtime.diagram.ui.render.util.CopyToImageUtil;
import org.eclipse.gmf.runtime.notation.Diagram;
import org.eclipse.papyrus.model2doc.core.builtintypes.ImageFormat;
import org.eclipse.papyrus.model2doc.core.generatorconfiguration.accessors.IOutputFileAccessor;
import org.eclipse.papyrus.model2doc.core.logger.IModel2DocLogger;


/**
 * Utils class to manipulate Image of GMF DIagrams
 */
public class GMFDiagramImageUtils {

	/**
	 * Generate image of diagram.
	 *
	 * @param diagram
	 * @param uri
	 * @param accessor
	 * @param margin
	 * @param format
	 * @param logger
	 */
	public static void generateImageOfDiagram(final Diagram diagram, final URI uri, IOutputFileAccessor accessor, final int margin, final ImageFormat format, final IModel2DocLogger logger) {
		final URL url = accessor.convertToURL(uri);
		final String pathRoot = accessor.urlToPathString(url, true);

		// we check all folders tree already exists, and we create them if not
		if (uri.segmentCount() > 1) {
			URI folderURI = uri.trimSegments(1);
			String folderPath = folderURI.toPlatformString(true);
			IFolder folder = ResourcesPlugin.getWorkspace().getRoot().getFolder(new Path(folderPath));

			createFolders(folder, logger);
		}

		final CopyToImageUtil copyImageUtil = new CustomCopyToImageUtils(margin);

		try {
			copyImageUtil.copyToImage(diagram, new Path(pathRoot), getFileFormat(format), new NullProgressMonitor(),
					PreferencesHint.USE_DEFAULTS);
		} catch (CoreException e) {
			logger.error(e);
		}

	}

	/**
	 * If the folder does not exist we create it and its parents if necessary
	 *
	 * @param folder
	 *            the folder to create
	 * @param logger
	 *            the logger used to log the messages during the process
	 */
	private static void createFolders(final IContainer folder, final IModel2DocLogger logger) {
		if (false == folder.exists()) {
			try {
				if (false == folder.getParent().exists()) {
					createFolders(folder.getParent(), logger);
				}
				if (folder instanceof IFolder) {
					((IFolder) folder).create(true, true, new NullProgressMonitor());
				} else if (folder instanceof IProject) {
					((IProject) folder).create(new NullProgressMonitor());
				}
			} catch (CoreException e) {
				logger.error(e);
			}
		}
	}

	/**
	 * @param format
	 * @return
	 */
	private static ImageFileFormat getFileFormat(ImageFormat format) {
		switch (format) {
		case PNG:
			return ImageFileFormat.PNG;
		case SVG:
		default:
			return ImageFileFormat.SVG;
		}
	}

}
