/*****************************************************************************
 * Copyright (c) 2019, 2020, 2023, 2025 CEA LIST and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Initial API and implementation
 *  Pauline DEVILLE (CEA LIST) pauline.deville@cea.fr - Bug 569251
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Bug 582327
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Issue GL-9
 *****************************************************************************/

package org.eclipse.papyrus.model2doc.emf.template2structure.internal.mapping;

import java.util.Collections;
import java.util.List;
import java.util.ListIterator;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.papyrus.model2doc.core.logger.IModel2DocLogger;
import org.eclipse.papyrus.model2doc.emf.template2structure.mapping.AbstractTemplateToStructureMapper;
import org.eclipse.papyrus.model2doc.emf.template2structure.mapping.IMappingService;

/**
 * This class is in charge of the creation of the document structure following the description of the document structure template model
 * We must create a new service instance for each new mapping process, in order to be able to detect infinite loop
 *
 * TODO : implements infinite loop detection
 */
public final class TemplateToStructureMappingService implements IMappingService {

	/**
	 * The registered mappers
	 */
	private final List<AbstractTemplateToStructureMapper<?>> mappers;

	/**
	 *
	 * Constructor.
	 *
	 * @param mappers
	 *            the mappers to use for this service instance
	 */
	public TemplateToStructureMappingService(final List<AbstractTemplateToStructureMapper<?>> mappers) {
		this.mappers = mappers;
	}

	/**
	 *
	 * @param documentTemplateElement
	 *            the input element
	 * @param expectedReturnedEClass
	 *            the expected EClass for the result of the mapping
	 * @return
	 *         the contributor answering to this mapping, or <code>null</code> when not found
	 */
	private AbstractTemplateToStructureMapper<?> getContributorFor(final EObject documentTemplateElement, final Class<?> expectedReturnedEClass) {
		AbstractTemplateToStructureMapper<?> contributor = null;
		final ListIterator<AbstractTemplateToStructureMapper<?>> iter = this.mappers.listIterator();
		while (iter.hasNext() && contributor == null) {
			final AbstractTemplateToStructureMapper<?> current = iter.next();
			if (current.handlesInput(documentTemplateElement) && current.handlesExpectedOutput(expectedReturnedEClass)) {
				contributor = current;
			}
		}
		return contributor;
	}

	/**
	 * @param documentTemplateElement
	 *            the input element
	 * @param semanticModelElement
	 *            an element of the user model
	 * @param expectedReturnedClass
	 *            the expected EClass for the result of the mapping
	 * @param logger
	 *            the logger used to log the messages during the process
	 * @return
	 *         the collection of created object answering to the mapping request
	 */
	@Override
	public <T> List<T> map(final EObject documentTemplateElement, final EObject semanticModelElement, final Class<T> expectedReturnedClass, final IModel2DocLogger logger) {
		AbstractTemplateToStructureMapper<?> contributor = getContributorFor(documentTemplateElement, expectedReturnedClass);
		List<T> result = null;
		if (null != contributor) {
			result = contributor.map(this, documentTemplateElement, semanticModelElement, expectedReturnedClass, logger);
			if (null == result) {
				result = Collections.emptyList();
			}
		} else {
			result = Collections.emptyList();
		}

		return result;

	}
}
