/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package activity.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.stream.Collectors;

import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.lsat.common.graph.directed.editable.EdgFactory;
import org.eclipse.lsat.common.graph.directed.editable.Edge;
import org.eclipse.lsat.common.graph.directed.editable.Node;
import org.eclipse.lsat.common.graph.directed.editable.SourceReference;
import org.eclipse.lsat.common.graph.directed.editable.TargetReference;

import activity.Action;
import activity.Activity;
import activity.ActivityFactory;
import activity.Claim;
import activity.Dependency;
import activity.Release;
import activity.SyncBar;
import activity.impl.ActivityQueries;
import machine.IResource;
import machine.ResourceType;

public final class ActivityUtil {

    public static final String EXPAND_DELIMITER = "_";

    private ActivityUtil() {
        /* Empty */
    }

    public static void addDependency(Activity activity, Node source, Node target) {
        if (!edgeExists(activity, source, target)) {
            Dependency dependency = ActivityFactory.eINSTANCE.createDependency();
            SourceReference sourceRef = EdgFactory.eINSTANCE.createSourceReference();
            sourceRef.setNode(source);
            dependency.setSource(sourceRef);
            TargetReference targetRef = EdgFactory.eINSTANCE.createTargetReference();
            targetRef.setNode(target);
            dependency.setTarget(targetRef);
            activity.getEdges().add(dependency);
        }
    }

    private static boolean edgeExists(Activity activity, Node source, Node target) {
        return activity.getEdges().stream().anyMatch(e -> (e.getSource() == source && e.getTarget() == target));
    }

    public static SyncBar addSyncBar(Activity activity) {
        SyncBar syncBar = ActivityFactory.eINSTANCE.createSyncBar();
        syncBar.setName(getFirstFreeName("S", activity));
        activity.getNodes().add(syncBar);
        return syncBar;
    }

    public static Release addRelease(Activity activity, IResource resource) {
        if (getReleases(activity, resource).isEmpty()) {
            Release release = ActivityFactory.eINSTANCE.createRelease();
            release.setName(getFirstFreeName("R", activity));
            release.setResource(resource);
            activity.getNodes().add(release);
        }
        return getReleases(activity, resource).iterator().next();
    }

    public static Claim addClaim(Activity activity, IResource resource) {
        if (getClaims(activity, resource).isEmpty()) {
            Claim claim = ActivityFactory.eINSTANCE.createClaim();
            claim.setName(getFirstFreeName("C", activity));
            claim.setResource(resource);
            activity.getNodes().add(claim);
        }
        return getClaims(activity, resource).iterator().next();
    }

    /**
     * Inserts a SyncBar if there are multiple incoming or outgoing edges. The edges are rerouted
     */
    public static void insertSyncBars(Action action) {
        Activity activity = (Activity)action.getGraph();
        if (action.getIncomingEdges().size() > 1) {
            SyncBar syncBar = ActivityUtil.addSyncBar(activity);
            action.getIncomingEdges().stream().map(Edge::getSourceNode).forEach(s->addDependency(activity, s, syncBar));
            new ArrayList<>(action.getIncomingEdges()).forEach(ActivityUtil::delete);
            addDependency(activity, syncBar, action);
        }
        if (action.getOutgoingEdges().size() > 1) {
            SyncBar syncBar = ActivityUtil.addSyncBar(activity);
            action.getOutgoingEdges().stream().map(Edge::getTargetNode).forEach(s->addDependency(activity, syncBar, s));
            new ArrayList<>(action.getOutgoingEdges()).forEach(ActivityUtil::delete);
            addDependency(activity, action, syncBar);
        }
    }

    public static void delete(Node node) {
        for (Edge edge: node.getIncomingEdges()) {
            delete(edge);
        }
        for (Edge edge: node.getOutgoingEdges()) {
            delete(edge);
        }
        if (node.getGraph() != null) {
            node.getGraph().getNodes().remove(node);
        }
        EcoreUtil.delete(node);
    }

    public static void delete(Edge edge) {
        if (edge.getTarget() instanceof Edge) {
            edge.getGraph().getEdges().add((Edge)edge.getTarget());
        }
        if (null != edge.getEdge()) {
            TargetReference target = EdgFactory.eINSTANCE.createTargetReference();
            target.setNode(edge.getSource().getNode());
            edge.getEdge().setTarget(target);
        }
        // this is really needed to clean up all relations:
        edge.setTarget(null);
        edge.setSource(null);
        if (edge.getGraph() != null) {
            edge.getGraph().getEdges().remove(edge);
        }
        EcoreUtil.delete(edge);
    }

    public static String getFirstFreeName(String prefix, final Activity activity) {
        Collection<String> names = activity.getNodes().stream().map(n -> n.getName()).collect(Collectors.toSet());
        int i = 1;
        while (names.contains(prefix + i)) {
            i++;
        }
        return prefix + i;
    }

    public static String getPassiveName(IResource resource) {
        if (resource.getResource().getResourceType() == ResourceType.PASSIVE) {
            return resource.getName();
        }
        return ResourceType.PASSIVE.getName() + '_' + resource.fqn().replace('.', '_');
    }

    /**
     * Get claims for a certain resource (typically 1)
     */
    public static Collection<Claim> getClaims(Activity activity, IResource resource) {
        return ActivityQueries.getActionsFor(resource, Claim.class, activity.getNodes()).asList();
    }

    /**
     * Get releases for a certain resource (typically 1)
     */
    public static Collection<Release> getReleases(Activity activity, IResource resource) {
        return ActivityQueries.getActionsFor(resource, Release.class, activity.getNodes()).asList();
    }
}
