/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.scheduler;

import java.io.IOException;
import java.io.Writer;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.eclipse.emf.common.util.EMap;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.lsat.common.graph.directed.Node;
import org.eclipse.lsat.common.scheduler.graph.Task;

import dispatching.Attribute;
import dispatching.Dispatch;
import dispatching.HasUserAttributes;
import lsat_graph.DispatchGraph;

/**
 *
 */
public class CsvWriter {
    private static final String NL = System.lineSeparator();

    private CsvWriter() {
        // helper class with static methods only
    }

    public static <T extends Task> void write(String name, Writer out, Map<T, CollectedScheduleData> results)
            throws IOException
    {
        writeHeader(name, out, results.keySet());
        writeData(out, results.values());
    }

    private static <T extends Task> void writeHeader(String name, Writer out, Collection<T> scheduledTasks)
            throws IOException
    {
        out.append("# Dispatch sequence ").append(name).append(NL);
        out.append("#TASKS").append(NL);
        var iter = scheduledTasks.iterator();
        for (int nodeIndex = 0; nodeIndex < scheduledTasks.size(); nodeIndex++) {
            var task = iter.next();
            var attr = getUserAttributes(task);
            if (nodeIndex == 0) { // first write header
                var metaHeader = List.of("nodeIndex", "task", "type");
                var metaHeaderAsStream = Stream.concat(metaHeader.stream(), attr.keySet().stream())
                        .collect(Collectors.joining(",", "## ", NL));
                out.append(metaHeaderAsStream);
            }
            out.append(Stream.concat(Stream.of(Integer.toString(nodeIndex), task.getName(), task.eClass().getName()), attr.values().stream())
                    .collect(Collectors.joining(",", "## ", NL)));
        }
        out.append("#DATA").append(NL);
        var header = List.of("step", "nodeIndex", "start time", "duration", "critical");
        var headerAsStream = header.stream().collect(Collectors.joining(",", "", NL));
        out.append(headerAsStream);
    }

    private static void writeData(Writer out, Collection<CollectedScheduleData> collectedData) throws IOException {
        if (collectedData.isEmpty()) {
            return;
        }
        for (int sample = 0; sample < collectedData.iterator().next().getSampleLength(); sample++) {
            var iter = collectedData.iterator();
            for (int nodeIndex = 0; nodeIndex < collectedData.size(); nodeIndex++) {
                var c = iter.next();
                out.append(String.join(",", Integer.toString(sample), Integer.toString(nodeIndex),
                        Double.toString(c.startTime[sample]), Double.toString(c.duration[sample]),
                        c.critical[sample] != 1 ? "0" : "1")).append(NL);
            }
        }
    }

    private static <T extends Task> Map<String, String> getUserAttributes(T task) {
        final Map<String, String> map = new LinkedHashMap<String, String>();
        Dispatch dispatch = getDispatch(task);
        EMap<Attribute, String> userAttributes = null;
        if (dispatch != null) {
            userAttributes = dispatch.getUserAttributes();
        }
        if (userAttributes != null) {
            addUserAttributes(dispatch, map);
        }
        return map;
    }

    private static <T extends Node> Dispatch getDispatch(final T node) {
        var graph = node.getGraph();
        if ((graph instanceof DispatchGraph dGraph)) {
            return dGraph.getDispatch();
        }
        return null;
    }

    private static void addUserAttributes(final EObject object, final Map<String, String> map) {
        if ((object != null)) {
            addUserAttributes(object.eContainer(), map);
        }
        if (object instanceof HasUserAttributes hasUserAttributes) {
            hasUserAttributes.getUserAttributes().forEach(e -> map.put(e.getKey().getName(), e.getValue()));
        }
    }
}
