/*
 * Copyright (c) 2007, 2018 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0, which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the
 * Eclipse Public License v. 2.0 are satisfied: GNU General Public License,
 * version 2 with the GNU Classpath Exception, which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 */

/*
 * $Id$
 */

package com.sun.ts.tests.jaxws.wsi.j2w.rpc.literal.R2305;

import java.util.ArrayList;
import java.util.Properties;
import java.util.StringTokenizer;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.sun.javatest.Status;
import com.sun.ts.tests.jaxws.sharedclients.ClientFactory;
import com.sun.ts.tests.jaxws.sharedclients.SOAPClient;
import com.sun.ts.tests.jaxws.sharedclients.simpleclient.*;
import com.sun.ts.tests.jaxws.wsi.constants.DescriptionConstants;
import com.sun.ts.tests.jaxws.wsi.utils.DescriptionUtils;
import com.sun.ts.lib.harness.*;

public class Client extends ServiceEETest implements DescriptionConstants {
  /**
   * The client.
   */
  private SOAPClient client;

  static SimpleTest service = null;

  /**
   * The document.
   */
  private Document document;

  /**
   * Test entry point.
   * 
   * @param args
   *          the command-line arguments.
   */
  public static void main(String[] args) {
    Client test = new Client();
    Status status = test.run(args, System.out, System.err);
    status.exit();
  }

  /**
   * @class.testArgs: -ap jaxws-url-props.dat
   * @class.setup_props: webServerHost; webServerPort; platform.mode;
   *
   * @param args
   * @param properties
   *
   * @throws Fault
   */
  public void setup(String[] args, Properties properties) throws Fault {
    client = ClientFactory.getClient(SimpleTestClient.class, properties, this,
        service);
    logMsg("setup ok");
  }

  public void cleanup() {
    logMsg("cleanup");
  }

  /**
   * @testName: testParameterOrder
   *
   * @assertion_ids: WSI:SPEC:R2305
   *
   * @test_Strategy: Retrieve the WSDL, generated by the Java-to-WSDL tool, and
   *                 examine the operation(s) making sure they have correct
   *                 parameterOrder attributes - if present.
   *
   * @throws Fault
   */
  public void testParameterOrder() throws Fault {
    document = client.getDocument();
    Element[] portTypes = DescriptionUtils.getPortTypes(document);
    for (int i = 0; i < portTypes.length; i++) {
      verifyPortType(portTypes[i]);
    }
  }

  protected void verifyPortType(Element element) throws Fault {
    Element[] children = DescriptionUtils.getChildElements(element,
        WSDL_NAMESPACE_URI, WSDL_OPERATION_LOCAL_NAME);
    for (int i = 0; i < children.length; i++) {
      verifyOperation(children[i]);
    }
  }

  protected void verifyOperation(Element element) throws Fault {
    Element output = getOutput(element);
    if (output == null) {
      return;
    }
    String parameterOrder = element.getAttribute(WSDL_PARAMETERORDER_ATTR)
        .trim();
    if (parameterOrder.length() == 0) {
      return;
    }
    String operationName = element.getAttribute(WSDL_NAME_ATTR);
    Element input = getInput(element);
    Element inputMessage = null;
    if (input != null) {
      inputMessage = getMessage(input);
    }
    Element outputMessage = getMessage(output);
    ArrayList inputNames;
    if (inputMessage != null) {
      inputNames = getPartNames(inputMessage);
    } else {
      inputNames = new ArrayList();
    }
    ArrayList outputNames = getPartNames(outputMessage);
    StringTokenizer tokenizer = new StringTokenizer(parameterOrder, " ");
    while (tokenizer.hasMoreTokens()) {
      String name = tokenizer.nextToken();
      if ((!inputNames.contains(name)) && (!outputNames.contains(name))) {
        throw new Fault("Parameter '" + name + "' of operation '"
            + operationName + "' not a part of in- or output message");
      }
      inputNames.remove(name);
      outputNames.remove(name);
    }
    if (outputNames.size() > 1) {
      throw new Fault("Parameter order of operation '" + operationName
          + "' leaves out more than 1 output message part (BP-R2305)");
    }
  }

  protected Element getOutput(Element element) throws Fault {
    Element[] children = DescriptionUtils.getChildElements(element,
        WSDL_NAMESPACE_URI, WSDL_OUTPUT_LOCAL_NAME);
    if (children.length > 0) {
      return children[0];
    }
    return null;
  }

  protected Element getInput(Element element) throws Fault {
    Element[] children = DescriptionUtils.getChildElements(element,
        WSDL_NAMESPACE_URI, WSDL_INPUT_LOCAL_NAME);
    if (children.length > 0) {
      return children[0];
    }
    return null;
  }

  protected Element getMessage(Element element) throws Fault {
    String message = element.getAttribute(DescriptionUtils.WSDL_MESSAGE_ATTR);
    int index = message.indexOf(':');
    if (index == -1) {
      throw new Fault(
          "In- or output element refers to unqualified message name '" + message
              + "' (BP-R2305)");
    }
    String localName = message.substring(index + 1);
    Element[] children = DescriptionUtils.getChildElements(document,
        WSDL_NAMESPACE_URI, WSDL_MESSAGE_LOCAL_NAME);
    for (int i = 0; i < children.length; i++) {
      String name = children[i].getAttribute(WSDL_NAME_ATTR);
      if (name.equals(localName)) {
        return children[i];
      }
    }
    throw new Fault("Message '" + localName + "' not found (BP-R2305)");
  }

  protected ArrayList getPartNames(Element element) throws Fault {
    Element[] children = DescriptionUtils.getChildElements(element,
        WSDL_NAMESPACE_URI, WSDL_PART_LOCAL_NAME);
    ArrayList names = new ArrayList(children.length);
    for (int i = 0; i < children.length; i++) {
      String name = children[i].getAttribute(WSDL_NAME_ATTR);
      if (names.contains(name)) {
        throw new Fault("Duplicate part name '" + name
            + "' encountered on message (BP-R2305)");
      }
      names.add(name);
    }
    return names;
  }
}
