//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2025, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.common;

import org.eclipse.escet.cif.metamodel.cif.Specification;
import org.eclipse.escet.cif.metamodel.java.CifConstructors;

/** CIF partial specification annotations utilities. */
public class CifPartialSpecAnnotationUtils {
    /** The name of the partial specification annotation. */
    public static final String NAME = "partial";

    /** Constructor for the {@link CifPartialSpecAnnotationUtils} class. */
    private CifPartialSpecAnnotationUtils() {
        // Static class.
    }

    /**
     * Returns whether the given specification is a partial specification.
     *
     * @param spec The specification.
     * @return {@code true} if the specification is a partial specification, {@code false} otherwise.
     */
    public static boolean isPartial(Specification spec) {
        return CifAnnotationUtils.hasAnnotation(spec, NAME);
    }

    /**
     * Make the given specification a partial specification, by adding a partial specification annotation if it does not
     * yet have one.
     *
     * @param spec The specification.
     */
    public static void makePartial(Specification spec) {
        if (!isPartial(spec)) {
            spec.getAnnotations().add(CifConstructors.newAnnotation(null, NAME, null));
        }
    }

    /**
     * Make the given specification a non-partial specification, by removing any partial specification annotations if
     * has.
     *
     * @param spec The specification.
     */
    public static void makeNonPartial(Specification spec) {
        CifAnnotationUtils.removeAnnotations(spec, NAME);
    }

    /**
     * Modify the given specification to be a partial or non-partial specification, as indicated, if it is not already
     * so.
     *
     * @param partial Whether to make the specification partial ({@code true}) or non-partial ({@code false}).
     * @param spec The specification.
     */
    public static void modifyPartial(Specification spec, boolean partial) {
        if (partial) {
            makePartial(spec);
        } else {
            makeNonPartial(spec);
        }
    }
}
